/* Copyright Airship and Contributors */

#import <Foundation/Foundation.h>
#import "UAAPIClient.h"
#import "UAPreferenceDataStore+Internal.h"
#import "UARemoteDataPayload+Internal.h"
#import "UARequestSession.h"
#import "UALocaleManager.h"

NS_ASSUME_NONNULL_BEGIN

///---------------------------------------------------------------------------------------
/// @name Typedefs
///---------------------------------------------------------------------------------------

/**
 * A block called when the refresh of the remote data succeeded.
 *
 * @param statusCode The request status code.
 * @param remoteData The refreshed remote data.
 */
typedef void (^UARemoteDataRefreshSuccessBlock)(NSUInteger statusCode, NSArray<NSDictionary *> * _Nullable remoteData);

/**
 * A block called when the refresh of the remote data failed.
 */
typedef void (^UARemoteDataRefreshFailureBlock)(void);

@interface UARemoteDataAPIClient : UAAPIClient

///---------------------------------------------------------------------------------------
/// @name Internal Methods
///---------------------------------------------------------------------------------------

/**
 * Create the remote data API client.
 *
 * @param config The Airship config.
 * @param dataStore A UAPreferenceDataStore to store persistent preferences
 * @param localeManager A UALocaleManager.
 * @return The remote data API client instance.
 */
+ (UARemoteDataAPIClient *)clientWithConfig:(UARuntimeConfig *)config dataStore:(UAPreferenceDataStore *)dataStore localeManager:(UALocaleManager *)localeManager;


/**
 * Create the remote data API client. Used for testing.
 *
 * @param config The Airship config.
 * @param dataStore A UAPreferenceDataStore to store persistent preferences
 * @param session The UARequestSession session.
 * @param localeManager A UALocaleManager.
 * @return The remote data API client instance.
 */
+ (UARemoteDataAPIClient *)clientWithConfig:(UARuntimeConfig *)config
                                  dataStore:(UAPreferenceDataStore *)dataStore
                                    session:(UARequestSession *)session
                              localeManager:(UALocaleManager *)localeManager;

/**
 * Refresh the remote data by calling the remote data cloud API.
 *
 * @param successBlock The block called when the refresh of the remote data succeeds.
 * @param failureBlock The block called when the refresh of the remote data fails.
 * @return A UADisposable token which can be used to cancel callback execution.
 *
 * Note: one block and only one block will be called.
 */
- (UADisposable *)fetchRemoteData:(UARemoteDataRefreshSuccessBlock)successBlock
                        onFailure:(UARemoteDataRefreshFailureBlock)failureBlock;

/**
 * Clears the last modified time for message list requests.
 */
- (void)clearLastModifiedTime;

@end

NS_ASSUME_NONNULL_END
